<?php
class Cart
{
    /** Ensure session cart store exists and return it by reference */
    private static function &store(): array {
        if (session_status() === PHP_SESSION_NONE) session_start();
        if (!isset($_SESSION['cart']) || !is_array($_SESSION['cart'])) $_SESSION['cart'] = [];
        return $_SESSION['cart'];
    }

    /** Build a cart key: "pid" or "pid:vid" */
    public static function key(int $productId, ?int $variantId = null): string {
        $productId = max(0, $productId);
        if ($variantId && $variantId > 0) return $productId . ':' . $variantId;
        return (string)$productId;
    }

    /** Parse a cart key back to [productId, variantId|null] */
    public static function parseKey(string $key): array {
        $key = trim($key);
        if ($key === '') return [0, null];
        if (strpos($key, ':') !== false) {
            [$p, $v] = explode(':', $key, 2);
            $p = (int)$p; $v = (int)$v;
            return [$p ?: 0, $v ?: null];
        }
        return [(int)$key, null];
    }

    /** Add quantity to an item (creates if missing) */
    public static function add(int $productId, int $qty = 1, ?int $variantId = null): void {
        $qty = max(1, $qty);
        $key = self::key($productId, $variantId);
        $s = &self::store();
        if (!isset($s[$key])) $s[$key] = 0;
        $s[$key] += $qty;
        if ($s[$key] <= 0) unset($s[$key]);
    }

    /** Set absolute quantity for a cart line (key may be "pid" or "pid:vid") */
    public static function set(string|int $keyOrPid, int $qty): void {
        $key = (string)$keyOrPid;
        $qty = (int)$qty;
        $s = &self::store();
        if ($qty <= 0) { unset($s[$key]); return; }
        $s[$key] = $qty;
    }

    /** Remove a line from the cart (key may be "pid" or "pid:vid") */
    public static function remove(string|int $keyOrPid): void {
        $key = (string)$keyOrPid;
        $s = &self::store();
        unset($s[$key]);
    }

    /** Get all cart lines as [cart_key => qty] */
    public static function items(): array {
        $s = &self::store();
        foreach ($s as $k => $q) {
            if (!is_int($q) || $q <= 0) unset($s[$k]);
        }
        return $s;
    }

    /** Empty the cart */
    public static function clear(): void {
        $s = &self::store();
        $s = [];
    }
}
