<?php
declare(strict_types=1);

class SessionDB implements SessionHandlerInterface
{
    private PDO $pdo;
    private string $table = 'sessions';
    private int $ttl;

    public function __construct(PDO $pdo, int $ttlSeconds = 604800) // 7 days
    {
        $this->pdo = $pdo;
        $this->ttl = $ttlSeconds;

        // Safer cookie defaults
        ini_set('session.use_strict_mode', '1');
        ini_set('session.cookie_httponly', '1');
        ini_set('session.use_only_cookies', '1');
        ini_set('session.use_trans_sid', '0');
        // Only set secure flag if HTTPS/Onion via HTTPS terminator
        ini_set('session.cookie_secure', (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? '1' : '0');
        ini_set('session.cookie_samesite', 'Lax');
    }

    /* ---- Required interface methods with PHP 8+ signatures ---- */

    public function open(string $path, string $name): bool
    {
        return true;
    }

    public function close(): bool
    {
        return true;
    }

    public function read(string $id): string
    {
        $sql = "SELECT data FROM {$this->table} WHERE id = ? AND expires >= ?";
        $st  = $this->pdo->prepare($sql);
        $st->execute([$id, time()]);
        $row = $st->fetch(PDO::FETCH_ASSOC);
        // Must return a string; empty string means “no data”
        return $row ? (string)$row['data'] : '';
    }

    public function write(string $id, string $data): bool
    {
        $exp = time() + $this->ttl;
        $sql = "INSERT INTO {$this->table} (id, data, expires)
                VALUES (?, ?, ?)
                ON DUPLICATE KEY UPDATE data = VALUES(data), expires = VALUES(expires)";
        $st  = $this->pdo->prepare($sql);
        return $st->execute([$id, $data, $exp]);
    }

    public function destroy(string $id): bool
    {
        $st = $this->pdo->prepare("DELETE FROM {$this->table} WHERE id = ?");
        return $st->execute([$id]);
    }

    public function gc(int $max_lifetime): int|false
    {
        $st = $this->pdo->prepare("DELETE FROM {$this->table} WHERE expires < ?");
        $st->execute([time()]);
        return $st->rowCount();
    }
}
