<?php
class Mailer
{
    private App $app;
    public function __construct(App $app) { $this->app = $app; }

    public function send(string $toEmail, string $toName, string $subject, string $html, ?string $alt = null): bool
    {
        $cfg = $this->app->cfg['mail'] ?? [];
        $fromEmail = $cfg['from_email'] ?? 'no-reply@localhost';
        $fromName  = $cfg['from_name']  ?? 'Store';
        $alt ??= strip_tags(preg_replace('~<br\s*/?>~i', "\n", $html));

        // Prefer PHPMailer if available
        if (class_exists('\PHPMailer\PHPMailer\PHPMailer')) {
            $mail = new \PHPMailer\PHPMailer\PHPMailer(true);
            try {
                $driver = strtolower($cfg['driver'] ?? 'smtp');
                if ($driver === 'smtp') {
                    $mail->isSMTP();
                    $mail->Host       = (string)($cfg['host'] ?? '');
                    $mail->Port       = (int)($cfg['port'] ?? 587);
                    $mail->SMTPAuth   = !empty($cfg['username']);
                    if ($mail->SMTPAuth) {
                        $mail->Username = (string)$cfg['username'];
                        $mail->Password = (string)$cfg['password'];
                    }
                    $enc = strtolower((string)($cfg['encryption'] ?? ''));
                    if ($enc === 'ssl') {
                        $mail->SMTPSecure = \PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_SMTPS;
                    } elseif ($enc === 'tls') {
                        $mail->SMTPSecure = \PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS;
                    }
                }

                // $mail->SMTPDebug = 2; // uncomment to debug
                // $mail->Debugoutput = function($str,$lvl){ error_log("SMTP[$lvl] $str"); };

                $mail->CharSet  = 'UTF-8';
                $mail->setFrom($fromEmail, $fromName);
                $mail->addAddress($toEmail, $toName);
                if (!empty($cfg['reply_to'])) {
                    $mail->addReplyTo($cfg['reply_to'], $cfg['reply_to_name'] ?? $fromName);
                }
                $mail->Subject = $subject;
                $mail->isHTML(true);
                $mail->Body    = $html;
                $mail->AltBody = $alt;

                $ok = $mail->send();
                if ($ok) return true;

                // If send() returned false without throwing, log & fall back.
                error_log('Mailer PHPMailer send() failed: ' . $mail->ErrorInfo);
            } catch (\Throwable $e) {
                error_log('Mailer PHPMailer error: ' . $e->getMessage());
                // fall through to mail()
            }
        }

        // Fallback: native mail()
        $headers  = "MIME-Version: 1.0\r\n";
        $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
        $headers .= 'From: ' . $fromName . ' <' . $fromEmail . ">\r\n";
        if (!empty($cfg['reply_to'])) {
            $headers .= 'Reply-To: ' . ($cfg['reply_to_name'] ?? $fromName) . ' <' . $cfg['reply_to'] . ">\r\n";
        }
        return @mail($toEmail, $subject, $html, $headers);
    }

    /* ---------- Convenience wrappers ---------- */

    public function sendOrderPlacedToCustomer(string $orderId, string $name, string $email, array $lines, string $currency, int $subtotalMinor): void
    {
        $html = $this->renderOrderSummaryHtml('We received your order', $orderId, $lines, $currency, $subtotalMinor);
        $this->send($email, $name, "Order $orderId received", $html);
    }

    public function sendOrderPlacedToAdmin(string $orderId, string $customerName, string $customerEmail, array $lines, string $currency, int $subtotalMinor): void
    {
        $to = $this->app->cfg['mail']['admin_email'] ?? '';
        if (!$to) return;
        $html = "<p><strong>New order received:</strong> {$orderId}</p>"
              . "<p><strong>Customer:</strong> ".htmlspecialchars($customerName)." &lt;".htmlspecialchars($customerEmail)."&gt;</p>"
              . $this->renderItemsTableHtml($lines, $currency, $subtotalMinor);
        $this->send($to, 'Admin', "New order $orderId", $html);
    }

    public function sendStatusChangedToCustomer(string $orderId, string $name, string $email, string $newStatus): void
    {
        $html = "<p>Your order <strong>".htmlspecialchars($orderId)."</strong> status is now: <strong>".htmlspecialchars(ucfirst($newStatus))."</strong>.</p>";
        $this->send($email, $name, "Order $orderId is $newStatus", $html);
    }

    public function sendStatusChangedToAdmin(string $orderId, string $newStatus): void
    {
        $to = $this->app->cfg['mail']['admin_email'] ?? '';
        if (!$to) return;
        $html = "<p>Order <strong>".htmlspecialchars($orderId)."</strong> status changed to <strong>".htmlspecialchars(ucfirst($newStatus))."</strong>.</p>";
        $this->send($to, 'Admin', "Order $orderId → $newStatus", $html);
    }

    /* ---------- Tiny rendering helpers ---------- */

    private function renderOrderSummaryHtml(string $title, string $orderId, array $lines, string $currency, int $subtotalMinor): string
    {
        $tbl = $this->renderItemsTableHtml($lines, $currency, $subtotalMinor);
        return "<p><strong>{$title}</strong></p>"
             . "<p>Order ID: <strong>".htmlspecialchars($orderId)."</strong></p>{$tbl}"
             . "<p>We’ll email you when the status changes.</p>";
    }

    private function renderItemsTableHtml(array $lines, string $currency, int $subtotalMinor): string
    {
        $cur = htmlspecialchars($currency);
        $rows = '';
        foreach ($lines as $l) {
            $name = htmlspecialchars($l['name'] ?? '');
            $qty  = (int)($l['qty'] ?? 1);
            $pr   = number_format(((int)($l['price_minor'] ?? 0))/100, 2);
            $tot  = number_format(((int)($l['price_minor'] ?? 0) * $qty)/100, 2);
            $rows .= "<tr><td>{$name}</td><td style='text-align:center'>{$qty}</td><td style='text-align:right'>{$cur} {$pr}</td><td style='text-align:right'>{$cur} {$tot}</td></tr>";
        }
        $sum = number_format($subtotalMinor/100, 2);
        return "<table style='border-collapse:collapse;width:100%'>
                  <thead>
                    <tr>
                      <th style='text-align:left;border-bottom:1px solid #ddd;padding:6px 0'>Product</th>
                      <th style='text-align:center;border-bottom:1px solid #ddd;padding:6px 0'>Qty</th>
                      <th style='text-align:right;border-bottom:1px solid #ddd;padding:6px 0'>Price</th>
                      <th style='text-align:right;border-bottom:1px solid #ddd;padding:6px 0'>Total</th>
                    </tr>
                  </thead>
                  <tbody>{$rows}</tbody>
                  <tfoot>
                    <tr>
                      <th colspan='3' style='text-align:right;padding-top:8px'>Subtotal</th>
                      <th style='text-align:right;padding-top:8px'>{$cur} {$sum}</th>
                    </tr>
                  </tfoot>
                </table>";
    }
}
