<?php

class BTCPay
{
    private App $app;

    public function __construct(App $app)
    {
        $this->app = $app;
    }

    /**
     * Create an invoice on BTCPay.
     *
     * @param string      $orderId      Your internal order ID
     * @param int         $amountMinor  Amount in minor units (e.g. cents)
     * @param string|null $redirectUrl  Where BTCPay should send the buyer back (success/cancel)
     */
    public function createInvoice(string $orderId, int $amountMinor, ?string $redirectUrl = null): array
    {
        $cfg     = $this->app->cfg['btcpay'] ?? [];
        $host    = rtrim($cfg['host'] ?? '', '/');
        $storeId = $cfg['storeId'] ?? '';
        $apiKey  = $cfg['apiKey'] ?? '';
        $currency= $cfg['currency'] ?? 'USD';

        $amount  = number_format($amountMinor / 100, 2, '.', '');

        if ($redirectUrl === null) {
            $redirectUrl = $this->app->baseUrl() . '/index.php?page=order_return&order=' . rawurlencode($orderId);
        }

        $payload = [
            'amount'   => $amount,
            'currency' => $currency,
            'metadata' => ['orderId' => $orderId],
            'checkout' => [
                'redirectURL'            => $redirectUrl,
                'redirectAutomatically'  => true,
            ],
        ];

        // Simulated mode if config isn't complete
        if (!$host || !$storeId || !$apiKey) {
            return [
                'status'       => 'simulated',
                'checkoutLink' => '/index.php?page=order_return&order=' . $orderId,
            ];
        }

        return $this->api('POST', "/api/v1/stores/{$storeId}/invoices", $payload);
    }

    /** Fetch a single invoice by its invoice ID. */
    public function getInvoice(string $invoiceId): array
    {
        $cfg     = $this->app->cfg['btcpay'] ?? [];
        $storeId = $cfg['storeId'] ?? '';
        if (!$storeId) throw new Exception('BTCPay storeId not configured.');
        return $this->api('GET', "/api/v1/stores/{$storeId}/invoices/{$invoiceId}");
    }

    /**
     * Verify BTCPay webhook signature.
     * Header is "BTCPay-Sig: sha256=<hex>".
     */
    public static function verifyWebhookSignature(string $rawBody, string $headerSig, string $secret): bool
    {
        if ($secret === '' || $headerSig === '') return false;
        if (!preg_match('/^sha256=([a-f0-9]{64})$/i', $headerSig, $m)) return false;
        $expected = hash_hmac('sha256', $rawBody, $secret);
        return hash_equals(strtolower($m[1]), strtolower($expected));
    }

    /**
     * Map BTCPay event+status to our local statuses.
     * Returns one of: 'paid', 'cancelled', 'pending'
     */
    public static function mapRemoteToLocalStatus(?string $eventType, ?string $remoteStatus): string
    {
        $e = strtolower((string)$eventType);
        $s = strtolower((string)$remoteStatus);

        // Event-first mapping (most reliable)
        if (in_array($e, ['invoicesettled','invoicecompleted','invoicepaid','invoiceconfirmed'], true)) return 'paid';
        if (in_array($e, ['invoiceexpired','invoiceinvalid','invoicecancelled','invoicecanceled'], true)) return 'cancelled';
        if ($e === 'invoiceprocessing' || $e === 'invoicecreated') return 'pending';

        // Fallback to reported status
        if (in_array($s, ['settled','complete','completed','confirmed','paid'], true)) return 'paid';
        if (in_array($s, ['expired','invalid','cancelled','canceled'], true)) return 'cancelled';
        return 'pending';
    }

    /**
     * Low-level API helper.
     */
    private function api(string $method, string $path, array $body = []): array
    {
        $cfg    = $this->app->cfg['btcpay'] ?? [];
        $host   = rtrim($cfg['host'] ?? '', '/');
        $apiKey = $cfg['apiKey'] ?? '';

        if (!$host)  throw new Exception('BTCPay host not configured.');
        if (!$apiKey) throw new Exception('BTCPay apiKey not configured.');

        $url = $host . $path;

        $ch = curl_init($url);
        $headers = [
            'Content-Type: application/json',
            'Authorization: token ' . $apiKey,
            'User-Agent: NoJSShop/1.0',
        ];

        $opts = [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER     => $headers,
            CURLOPT_TIMEOUT        => 25,
        ];

        $method = strtoupper($method);
        if ($method !== 'GET') {
            $opts[CURLOPT_CUSTOMREQUEST] = $method;
            $opts[CURLOPT_POSTFIELDS]    = json_encode($body);
        }

        curl_setopt_array($ch, $opts);

        $res = curl_exec($ch);
        if ($res === false) {
            $err = curl_error($ch);
            curl_close($ch);
            throw new Exception('BTCPay HTTP error: ' . $err);
        }

        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $json = json_decode($res, true);

        if ($code >= 400) {
            $msg = is_array($json) ? ($json['message'] ?? $res) : $res;
            throw new Exception('BTCPay error: ' . $code . ' ' . $msg);
        }

        return is_array($json) ? $json : [];
    }
}
