<?php
//ini_set('display_errors',1);
//error_reporting(E_ALL);
$autoload = __DIR__ . '/../vendor/autoload.php';
if (is_file($autoload)) { require $autoload; }
require __DIR__.'/../src/App.php';
require __DIR__.'/../src/CSRF.php';
require __DIR__.'/../src/Cart.php';
require __DIR__.'/../src/BTCPay.php';
require __DIR__.'/../src/Mailer.php';

/* ===================== PGP utils (no-JS) ===================== */
function pgp_clean_public(string $raw): string {
    $raw = html_entity_decode($raw, ENT_QUOTES|ENT_HTML5, 'UTF-8');
    $raw = str_replace("\r", "", trim($raw));
    if (strpos($raw, "\\n") !== false && strpos($raw, "\n") === false) {
        $raw = str_replace("\\n", "\n", $raw);
    }
    if (preg_match('/-----BEGIN PGP PUBLIC KEY BLOCK-----.*?-----END PGP PUBLIC KEY BLOCK-----/s', $raw, $m)) {
        return $m[0];
    }
    return $raw;
}
function shell_exec_available(): bool {
    $disabled = ini_get('disable_functions') ?: '';
    if (stripos($disabled, 'shell_exec') !== false) return false;
    return function_exists('shell_exec');
}
function find_gpg(): ?string {
    if (!shell_exec_available()) return null;
    $p = trim((string)@shell_exec('command -v gpg 2>/dev/null'));
    return $p !== '' ? $p : null;
}
/** Encrypt $plaintext with ASCII-armored output. Prefers php-gnupg, falls back to gpg CLI. */
function pgp_encrypt_armored(string $publicKey, string $plaintext, ?string &$err = null): ?string {
    $publicKey = pgp_clean_public($publicKey);

    if (extension_loaded('gnupg')) {
        $tmp = sys_get_temp_dir() . '/gnuhome-' . bin2hex(random_bytes(6));
        if (!@mkdir($tmp, 0700)) { $err = 'Cannot create temp GnuPG home.'; return null; }
        $oldHome = getenv('GNUPGHOME');
        putenv('GNUPGHOME=' . $tmp);
        try {
            $g = new gnupg();
            if (method_exists($g, 'seterrormode')) {
                $mode = defined('GNUPG_ERROR_EXCEPTION') ? GNUPG_ERROR_EXCEPTION : 2;
                $g->seterrormode($mode);
            }
            if (method_exists($g, 'setarmor')) $g->setarmor(1);
            $info = $g->import($publicKey);
            $fpr = '';
            if (is_array($info)) {
                if (!empty($info['fingerprint'])) $fpr = (string)$info['fingerprint'];
                elseif (!empty($info['imports'])) foreach ($info['imports'] as $imp) { if (!empty($imp['fingerprint'])) { $fpr = (string)$imp['fingerprint']; break; } }
                elseif (!empty($info['fingerprints'])) $fpr = (string)($info['fingerprints'][0] ?? '');
            }
            if ($fpr === '' && method_exists($g, 'keyinfo')) {
                $keys = $g->keyinfo('');
                if (is_array($keys)) {
                    foreach ($keys as $k) foreach (($k['subkeys'] ?? []) as $sk) {
                        $caps = (string)($sk['capabilities'] ?? '');
                        $canE = ($sk['can_encrypt'] ?? null) ? true : (strpos($caps, 'e') !== false);
                        if (!empty($sk['fingerprint']) && $canE) { $fpr = (string)$sk['fingerprint']; break 2; }
                    }
                    if ($fpr === '' && !empty($keys[0]['subkeys'][0]['fingerprint'])) $fpr = (string)$keys[0]['subkeys'][0]['fingerprint'];
                }
            }
            if ($fpr === '') { $err = 'PGP key import failed (no fingerprint).'; return null; }
            if (method_exists($g, 'clearencryptkeys')) $g->clearencryptkeys();
            $g->addencryptkey($fpr);
            $enc = $g->encrypt($plaintext);
            if (!$enc || strpos($enc, 'BEGIN PGP MESSAGE') === false) { $err = 'php-gnupg encryption failed.'; return null; }
            return $enc;
        } catch (Throwable $e) {
            $err = 'php-gnupg: '.$e->getMessage();
            return null;
        } finally {
            if ($oldHome === false) { putenv('GNUPGHOME'); } else { putenv('GNUPGHOME=' . $oldHome); }
            if (is_dir($tmp)) { foreach (glob($tmp.'/*') ?: [] as $f) @unlink($f); @rmdir($tmp); }
        }
    }

    if (!shell_exec_available()) { $err = 'PGP unavailable: no php-gnupg and shell_exec disabled.'; return null; }
    $gpgBin = find_gpg();
    if (!$gpgBin) { $err = 'PGP unavailable: gpg binary not found.'; return null; }

    $tmp = sys_get_temp_dir() . '/pgp-' . bin2hex(random_bytes(6));
    if (!@mkdir($tmp, 0700)) { $err = 'Cannot create temp keyring dir.'; return null; }

    try {
        $pub = $tmp.'/pub.asc'; $plain = $tmp.'/plain.txt'; $cipher = $tmp.'/cipher.asc';
        file_put_contents($pub, $publicKey);
        file_put_contents($plain, str_replace("\r", "", $plaintext));

        $outFpr = @shell_exec('LANG=C '.escapeshellcmd($gpgBin)
            .' --batch --homedir '.escapeshellarg($tmp)
            .' --with-colons --import-options show-only --import '.escapeshellarg($pub).' 2>&1');
        if (!preg_match('/^fpr:(?:[^:]*:){8}([0-9A-Fa-f]{40}):/m', (string)$outFpr, $m)) {
            $err = 'Could not parse key fingerprint from gpg.';
            return null;
        }
        $fpr = strtoupper($m[1]);

        @shell_exec('LANG=C '.escapeshellcmd($gpgBin).' --batch --homedir '.escapeshellarg($tmp)
            .' --import '.escapeshellarg($pub).' 2>&1');
        @shell_exec('LANG=C '.escapeshellcmd($gpgBin).' --batch --yes --homedir '.escapeshellarg($tmp)
            .' --trust-model always -a -r '.escapeshellarg($fpr)
            .' -o '.escapeshellarg($cipher).' -e '.escapeshellarg($plain).' 2>&1');

        if (!is_file($cipher)) { $err = 'gpg CLI encryption failed.'; return null; }
        $armored = file_get_contents($cipher);
        if (!$armored || strpos($armored, 'BEGIN PGP MESSAGE') === false) { $err = 'Invalid cipher output.'; return null; }
        return $armored;
    } finally {
        foreach ([$pub ?? null, $plain ?? null, $cipher ?? null] as $f) if ($f && is_file($f)) @unlink($f);
        if (is_dir($tmp)) { foreach (glob($tmp.'/*') ?: [] as $f) @unlink($f); @rmdir($tmp); }
    }
}
/** Get merchant PGP public key: config override or first admin with a key */
function get_orders_pgp_key(App $app): string {
    $k = $app->cfg['orders_pgp_public'] ?? '';
    if ($k) return $k;
    $row = $app->db->query("SELECT pgp_public FROM admin_users WHERE pgp_public<>'' ORDER BY id ASC LIMIT 1")->fetch();
    return $row['pgp_public'] ?? '';
}

/* =============== Variant helpers =============== */
function variant_label_from_json(?string $json): string {
    if (!$json) return '';
    $attrs = json_decode($json, true) ?: [];
    if (!$attrs) return '';
    $parts = [];
    foreach ($attrs as $k => $v) {
        $label = preg_replace('~^attribute_(pa_)?~', '', (string)$k);
        $label = str_replace(['-', '_'], ' ', $label);
        $parts[] = ucfirst($label) . ': ' . ucfirst((string)$v);
    }
    return implode(' · ', $parts);
}

$app = new App();
CSRF::bootstrap($app->cfg['csrf_secret'] ?? 'f0055bc2a3d32b0be2c9433c2fa03f555065c5bbbfc4b9b7fc93878df17974c2');
$page = $_GET['page'] ?? 'home';

/* ===================== Routes ===================== */
switch ($page) {

    case 'home':
        // Active categories that have at least one active product
        $cats = $app->db->query(
            "SELECT c.id, c.name, c.slug
             FROM categories c
             WHERE c.is_active=1
               AND EXISTS (SELECT 1 FROM products p WHERE p.category_id=c.id AND p.is_active=1)
             ORDER BY c.name"
        )->fetchAll();

        // Get top 3 products per category (newest first)
        $sections = [];
        $stmtTop3 = $app->db->prepare(
    "SELECT
        p.id,
        p.slug,
        p.name,
        p.price_minor,
        p.image,
        p.product_type,
        (
          SELECT MIN(v.price_minor)
          FROM product_variants v
          WHERE v.product_id = p.id AND v.is_active = 1
        ) AS min_variant_price_minor
     FROM products p
     WHERE p.is_active = 1 AND p.category_id = ?
     ORDER BY p.id DESC
     LIMIT 3"
);
        foreach ($cats as $c) {
            $stmtTop3->execute([$c['id']]);
            $sections[] = [
                'category' => $c,
                'products' => $stmtTop3->fetchAll(),
            ];
        }
        $app->render('home', ['sections' => $sections]);
        break;

    case 'category':
    $slug = trim($_GET['slug'] ?? '');
    $subSlug = isset($_GET['sub']) ? trim($_GET['sub']) : null;

    // 1) Load category
    $stmt = $app->db->prepare("SELECT id, name, slug FROM categories WHERE slug = ? AND is_active = 1");
    $stmt->execute([$slug]);
    $category = $stmt->fetch();
    if (!$category) { http_response_code(404); echo 'Category not found'; exit; }

    // 2) If subcategory slug provided, validate it belongs to this category
    $subcategory = null;
    $params = [(int)$category['id']];

    $whereSub = '';
    if ($subSlug !== null && $subSlug !== '') {
        $stmtS = $app->db->prepare("
            SELECT id, name, slug
            FROM subcategories
            WHERE slug = ? AND category_id = ? AND is_active = 1
            LIMIT 1
        ");
        $stmtS->execute([$subSlug, (int)$category['id']]);
        $subcategory = $stmtS->fetch();

        if (!$subcategory) {
            // You can choose to 404, or fall back to the whole category.
            http_response_code(404); echo 'Subcategory not found'; exit;
            // Or fallback:
            // $subcategory = null; // keep showing full category
        } else {
            $whereSub = " AND p.subcategory_id = ? ";
            $params[] = (int)$subcategory['id'];
        }
    }

    // 3) Load products (filtered by subcategory when present)
    $sql = "
        SELECT
            p.id,
            p.slug,
            p.name,
            p.price_minor,
            p.image,
            p.product_type,
            (
              SELECT MIN(v.price_minor)
              FROM product_variants v
              WHERE v.product_id = p.id AND v.is_active = 1
            ) AS min_variant_price_minor
        FROM products p
        WHERE p.is_active = 1
          AND p.category_id = ?
          $whereSub
        ORDER BY p.id DESC
    ";
    $stmtP = $app->db->prepare($sql);
    $stmtP->execute($params);
    $products = $stmtP->fetchAll();

    // 4) Render (also pass $subcategory for nicer heading, if you want)
    $app->render('category', compact('category','subcategory','products'));
    break;


    case 'product':
        $slug = $_GET['slug'] ?? '';
        $stmt = $app->db->prepare('SELECT * FROM products WHERE slug=? AND is_active=1');
        $stmt->execute([$slug]);
        $product = $stmt->fetch();
        if (!$product) { http_response_code(404); echo 'Not found'; exit; }

        // Load active variants for variable products
        $variants = [];
        if (($product['product_type'] ?? 'simple') === 'variable') {
            $v = $app->db->prepare('SELECT id, sku, attributes_json, price_minor, image, is_active
                                    FROM product_variants
                                    WHERE product_id=? AND is_active=1
                                    ORDER BY id ASC');
            $v->execute([(int)$product['id']]);
            $variants = $v->fetchAll();
        }
        $app->render('product', compact('product','variants'));
        break;

    case 'add_to_cart':
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') { http_response_code(405); exit; }
        CSRF::check();
        $pid = (int)($_POST['product_id'] ?? 0);
        $qty = (int)($_POST['qty'] ?? 1);
        $vid = isset($_POST['variant_id']) && $_POST['variant_id'] !== '' ? (int)$_POST['variant_id'] : null;

        // Validate product
        $stmt = $app->db->prepare('SELECT id, product_type, is_active FROM products WHERE id=?');
        $stmt->execute([$pid]);
        $prod = $stmt->fetch();
        if (!$prod || !$prod['is_active']) { http_response_code(400); echo 'Invalid product'; exit; }

        // If variable, variant is required & must belong to product
        if (($prod['product_type'] ?? 'simple') === 'variable') {
            if (!$vid) { http_response_code(400); echo 'Please select a variant'; exit; }
            $chk = $app->db->prepare('SELECT id FROM product_variants WHERE id=? AND product_id=? AND is_active=1');
            $chk->execute([$vid, $pid]);
            if (!$chk->fetch()) { http_response_code(400); echo 'Invalid variant'; exit; }
        } else {
            $vid = null;
        }

        Cart::add($pid, max(1,$qty), $vid);
        header('Location: '.$app->baseUrl().'/index.php?page=cart'); exit;

    case 'cart':
        // Resolve items (supports keys "pid" and "pid:vid")
        $entries  = Cart::items();          // [cart_key => qty]
        $details  = [];
        $subtotal = 0;

        if ($entries) {
            $pids = []; $vids = [];
            foreach ($entries as $key => $qty) {
                [$pid, $vid] = Cart::parseKey((string)$key);
                if ($pid) $pids[$pid] = true;
                if ($vid) $vids[$vid] = true;
            }

            // Products
            $mapP = [];
            if ($pids) {
                $in = implode(',', array_fill(0, count($pids), '?'));
                $stmt = $app->db->prepare("SELECT id, name, price_minor FROM products WHERE id IN ($in)");
                $stmt->execute(array_keys($pids));
                foreach ($stmt as $row) $mapP[$row['id']] = $row;
            }

            // Variants
            $mapV = [];
            if ($vids) {
                $in = implode(',', array_fill(0, count($vids), '?'));
                $stmt = $app->db->prepare("SELECT id, product_id, attributes_json, price_minor, image FROM product_variants WHERE id IN ($in)");
                $stmt->execute(array_keys($vids));
                foreach ($stmt as $row) $mapV[$row['id']] = $row;
            }

            foreach ($entries as $key => $qty) {
                [$pid, $vid] = Cart::parseKey((string)$key);
                if (!isset($mapP[$pid])) continue;

                $p = $mapP[$pid];
                $name = $p['name'];
                $priceMinor = (int)$p['price_minor'];

                if ($vid && isset($mapV[$vid])) {
                    $v = $mapV[$vid];
                    $priceMinor = (int)$v['price_minor'];
                    $label = variant_label_from_json($v['attributes_json']);
                    if ($label) $name .= ' (' . $label . ')';
                }

                $qty = max(1, (int)$qty);
                $lineTotal = $priceMinor * $qty;
                $subtotal += $lineTotal;

                $details[] = [
                    'cart_key'    => (string)$key, // for update form name="qty[cart_key]"
                    'product_id'  => $pid,
                    'variant_id'  => $vid,
                    'name'        => $name,
                    'qty'         => $qty,
                    'price_minor' => $priceMinor,
                    'total_minor' => $lineTotal,
                ];
            }
        }
        $app->render('cart', ['lines'=>$details, 'subtotal'=>$subtotal]);
        break;

    case 'update_cart':
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') { http_response_code(405); exit; }
        CSRF::check();
        // Accept both numeric product ids and "pid:vid" keys
        foreach (($_POST['qty'] ?? []) as $key => $qty) {
            $keyStr = (string)$key;
            Cart::set($keyStr, (int)$qty);
        }
        header('Location: '.$app->baseUrl().'/index.php?page=cart'); exit;

    case 'checkout':
        $items = Cart::items();
        if (!$items) { header('Location: '.$app->baseUrl().'/index.php?page=cart'); exit; }
        $app->render('checkout', []);
        break;

    case 'place_order':
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') { http_response_code(405); exit; }
        CSRF::check();

        $name  = trim($_POST['name'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $phone    = trim($_POST['phone'] ?? '');
		$street   = trim($_POST['address_street'] ?? '');
		$city     = trim($_POST['address_city'] ?? '');
		$postcode = trim($_POST['address_postcode'] ?? '');
        // Hidden combined address from the form (for compatibility),
    // or rebuild it from the parts if not present.
    $addr = trim($_POST['address'] ?? '');
    if ($addr === '') {
        $parts = array_filter([$street, $city, $postcode], static fn($v) => $v !== '');
        $addr  = implode("\n", $parts);
    }

    // ===== Validation =====
    // Require: name, email, phone, street, city
    if (!$name || !$email || !$phone || !$street || !$city) {
        http_response_code(400);
        echo 'Missing fields';
        exit;
    }
    // optional: very light email/phone sanity checks (don’t be too strict)
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        http_response_code(400); echo 'Invalid email'; exit;
    }
    if (strlen(preg_replace('/\D+/', '', $phone)) < 7) { // at least 7 digits
        http_response_code(400); echo 'Invalid phone'; exit;
    }
        // Recompute totals (variant-aware) from cart
        $entries = Cart::items();
        if (!$entries) { http_response_code(400); echo 'Cart empty'; exit; }

        $pids=[]; $vids=[];
        foreach ($entries as $key=>$qty) {
            [$pid,$vid] = Cart::parseKey((string)$key);
            if ($pid) $pids[$pid]=true;
            if ($vid) $vids[$vid]=true;
        }

        // Products
        $mapP=[];
        if ($pids) {
            $inP = implode(',', array_fill(0, count($pids), '?'));
            $stmt = $app->db->prepare("SELECT id, name, price_minor FROM products WHERE id IN ($inP)");
            $stmt->execute(array_keys($pids));
            foreach ($stmt as $row) $mapP[$row['id']] = $row;
        }

        // Variants
        $mapV=[];
        if ($vids) {
            $inV = implode(',', array_fill(0, count($vids), '?'));
            $stmtV = $app->db->prepare("SELECT id, product_id, attributes_json, price_minor FROM product_variants WHERE id IN ($inV)");
            $stmtV->execute(array_keys($vids));
            foreach ($stmtV as $row) $mapV[$row['id']] = $row;
        }

        $subtotal = 0;
        $resolved = []; // lines to insert into order_items
        foreach ($entries as $key=>$qty) {
            [$pid,$vid] = Cart::parseKey((string)$key);
            if (!isset($mapP[$pid])) continue;

            $p = $mapP[$pid];
            $priceMinor = (int)$p['price_minor'];
            $nameLine   = $p['name'];
            $attrsJson  = null;

            if ($vid && isset($mapV[$vid])) {
                $v = $mapV[$vid];
                $priceMinor = (int)$v['price_minor'];
                $attrsJson  = $v['attributes_json'];
                $lab = variant_label_from_json($attrsJson);
                if ($lab) $nameLine .= ' (' . $lab . ')';
            }

            $qty = max(1,(int)$qty);
            $lineTotal = $priceMinor * $qty;
            $subtotal += $lineTotal;

            $resolved[] = [
                'product_id'  => $pid,
                'variant_id'  => $vid,
                'name'        => $nameLine,
                'qty'         => $qty,
                'price_minor' => $priceMinor,
                'attributes_json' => $attrsJson,
            ];
        }

        if ($subtotal <= 0) { http_response_code(400); echo 'Invalid total'; exit; }

        // --- Encrypt Name+Address with PGP ---
        $ordersKey = get_orders_pgp_key($app);
        if (!$ordersKey) { http_response_code(500); echo 'Checkout unavailable: encryption key not configured.'; exit; }

        $pii = "Name: {$name}\nEmail: {$email}\nPhone: {$phone}\nAddress:\n{$addr}\n";
		$encErr = null;
		$encAddr = pgp_encrypt_armored($ordersKey, $pii, $encErr);
        if (!$encAddr) { http_response_code(500); echo 'Encryption error: '.htmlspecialchars($encErr ?: 'unknown'); exit; }

        // Persist order as pending
        $app->db->beginTransaction();
        $orderId = 'ORD-' . date('YmdHis') . '-' . bin2hex(random_bytes(3));
        $stmtO = $app->db->prepare('INSERT INTO orders (order_id, name, email, address, total_minor, currency, status, created_at) VALUES (?,?,?,?,?,?,?,NOW())');
        $stmtO->execute([$orderId, $name, $email, $encAddr, $subtotal, $app->cfg['btcpay']['currency'], 'pending']);

        $stmtItem = $app->db->prepare('
            INSERT INTO order_items (order_id, product_id, name, qty, price_minor, variant_id, attributes_json)
            VALUES (?,?,?,?,?,?,?)
        ');
        foreach ($resolved as $r) {
            $stmtItem->execute([
                $orderId,
                $r['product_id'],
                $r['name'],
                $r['qty'],
                $r['price_minor'],
                $r['variant_id'],
                $r['attributes_json'],
            ]);
        }
        $app->db->commit();

        // Create BTCPay invoice and remember invoice id
        $btcpay = new BTCPay($app);
        $returnUrl = $app->baseUrl().'/index.php?page=order_return&order='.$orderId;
        $inv = $btcpay->createInvoice($orderId, (int)$subtotal, $returnUrl);
        $invoiceId = $inv['id'] ?? null;
        if ($invoiceId) {
            $u = $app->db->prepare('UPDATE orders SET btcpay_invoice_id=? WHERE order_id=?');
            $u->execute([$invoiceId, $orderId]);
        }
		
		$mailer = new Mailer($app);
		$mailer->sendOrderPlacedToCustomer($orderId, $name, $email, $resolved, $app->cfg['btcpay']['currency'], (int)$subtotal);
		$mailer->sendOrderPlacedToAdmin($orderId, $name, $email, $resolved, $app->cfg['btcpay']['currency'], (int)$subtotal);


        $link = $inv['checkoutLink'] ?? ($inv['url'] ?? '');
        if (!$link) {
            header('Location: '.$app->baseUrl().'/index.php?page=order_return&order='.$orderId);
            exit;
        }

        Cart::clear();
        header('Location: '.$link);
        exit;

    case 'order_success':
        $order = $_GET['order'] ?? '';
        $app->render('order_success', ['order'=>$order]);
        break;

    case 'order_return':
        $orderId = trim($_GET['order'] ?? '');
        if ($orderId === '') { http_response_code(400); echo 'Missing order id'; exit; }

        $stmt = $app->db->prepare('SELECT order_id, btcpay_invoice_id, status FROM orders WHERE order_id=?');
        $stmt->execute([$orderId]);
        $order = $stmt->fetch();
        if (!$order) { http_response_code(404); echo 'Order not found'; exit; }

        $status = $order['status'];

        if (!empty($order['btcpay_invoice_id'])) {
            $btcpay = new BTCPay($app);
            try {
                $inv = $btcpay->getInvoice($order['btcpay_invoice_id']);
                $remote = strtolower($inv['status'] ?? ($inv['data']['status'] ?? ''));
                if (in_array($remote, ['settled','complete','completed','confirmed','paid'], true))      $status = 'paid';
                elseif (in_array($remote, ['expired','invalid','cancelled','canceled'], true))           $status = 'cancelled';
                else                                                                                    $status = 'pending';
                if ($status !== $order['status']) {
                    $u = $app->db->prepare('UPDATE orders SET status=? WHERE order_id=?');
                    $u->execute([$status, $orderId]);
                }
            } catch (Throwable $e) {
                // leave status as-is; optionally log $e->getMessage()
            }
        }

        if ($status === 'paid') {
            $app->render('order_success', ['order'=>$orderId]);
        } elseif ($status === 'cancelled') {
            $app->render('order_cancel', ['order'=>$orderId]);   // create templates/order_cancel.php
        } else {
            $app->render('order_pending', ['order'=>$orderId]);  // create templates/order_pending.php (optional)
        }
        break;
		
	case 'btcpay_webhook':
{
    // No CSRF here; BTCPay posts JSON
    $secret = (string)($app->cfg['btcpay']['webhookSecret'] ?? '');
    if ($secret === '') { http_response_code(501); echo 'Webhook not configured'; exit; }

    $raw = file_get_contents('php://input') ?: '';
    $sig = $_SERVER['HTTP_BTCPAY_SIG'] ?? ($_SERVER['HTTP_BTCPAY_SIGNATURE'] ?? '');

    // 1) Verify signature
    if (!BTCPay::verifyWebhookSignature($raw, $sig, $secret)) {
        http_response_code(401);
        echo 'Invalid signature';
        exit;
    }

    // 2) Parse JSON (BTCPay can send slightly different shapes across versions)
    $j = json_decode($raw, true);
    if (!is_array($j)) { http_response_code(400); echo 'Bad JSON'; exit; }

    $eventType = $j['type'] ?? ($j['event'] ?? null);
    $data      = $j['data'] ?? ($j['payload'] ?? $j);

    $invoiceId = $data['invoiceId'] ?? ($j['invoiceId'] ?? null);
    $remoteSt  = $data['status']    ?? ($j['status']    ?? null);
    $meta      = $data['metadata']  ?? [];

    $orderId   = $meta['orderId'] ?? null;

    // 3) If orderId is missing, try lookup by invoiceId
    if (!$orderId && $invoiceId) {
        $stmt = $app->db->prepare('SELECT order_id FROM orders WHERE btcpay_invoice_id=? LIMIT 1');
        $stmt->execute([$invoiceId]);
        $row = $stmt->fetch();
        if ($row) $orderId = $row['order_id'];
    }

    if (!$orderId) {
        http_response_code(200);
        echo json_encode(['ok'=>true, 'note'=>'no matching order']);
        exit;
    }

    // 4) Map remote -> local status (e.g., processing->pending, settled->paid, etc.)
    $local = BTCPay::mapRemoteToLocalStatus($eventType, $remoteSt);

    // 5) Update order if changed (store invoice id if provided)
    $stmt = $app->db->prepare('SELECT order_id, name, email, currency, total_minor, status, btcpay_invoice_id FROM orders WHERE order_id=?');
    $stmt->execute([$orderId]);
    $order = $stmt->fetch();
    if ($order) {
        $statusChanged = ($local !== $order['status']);
        $newInvoiceId  = $order['btcpay_invoice_id'] ?: ($invoiceId ?? null);

        if ($statusChanged || ($newInvoiceId && $newInvoiceId !== $order['btcpay_invoice_id'])) {
            $u = $app->db->prepare('UPDATE orders SET status=?, btcpay_invoice_id=COALESCE(?, btcpay_invoice_id) WHERE order_id=?');
            $u->execute([$local, $newInvoiceId, $orderId]);

            // ---- Email notifications on status change ----
            if ($statusChanged) {
                try {
                    $mailer = new Mailer($app);

                    // Customer notice
                    $mailer->sendStatusChangedToCustomer(
                        $order['order_id'],
                        (string)$order['name'],
                        (string)$order['email'],
                        (string)$local
                    );

                    // Admin notice
                    $mailer->sendStatusChangedToAdmin($order['order_id'], (string)$local);
                } catch (Throwable $e) {
                    // Don’t fail the webhook; just log.
                    error_log('Webhook mail error: ' . $e->getMessage());
                }
            }
        }
    }

    http_response_code(200);
    header('Content-Type: application/json');
    echo json_encode(['ok'=>true, 'orderId'=>$orderId, 'status'=>$local]);
    exit;
}



    default:
        http_response_code(404);
        echo 'Not found';
}
