<?php
//ini_set('display_errors', '1');
//ini_set('display_startup_errors', '1');
//error_reporting(E_ALL);
$autoload = __DIR__ . '/../vendor/autoload.php';
if (is_file($autoload)) { require $autoload; }
require __DIR__ . '/../src/Mailer.php';
/* Show ALL fatals on the page, even parse/compile errors */
set_error_handler(function ($severity, $message, $file, $line) {
    if (!(error_reporting() & $severity)) { return false; }
    throw new ErrorException($message, 0, $severity, $file, $line);
});
set_exception_handler(function ($e) {
    http_response_code(500);
    echo "<pre>Uncaught ".get_class($e).": ".$e->getMessage()."\nIn ".$e->getFile().":".$e->getLine()."\n\n".$e->getTraceAsString()."</pre>";
});
register_shutdown_function(function () {
    $e = error_get_last();
    if ($e && in_array($e['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        http_response_code(500);
        echo "<pre>FATAL: {$e['message']}\nIn {$e['file']}:{$e['line']}</pre>";
    }
});

require __DIR__ . '/../src/App.php';
require __DIR__ . '/../src/CSRF.php';
require __DIR__ . '/../src/AdminAuth.php';
require __DIR__ . '/../src/BTCPay.php';

$app = new App();
CSRF::bootstrap($app->cfg['csrf_secret'] ?? 'f0055bc2a3d32b0be2c9433c2fa03f555065c5bbbfc4b9b7fc93878df17974c2');
$action = $_GET['action'] ?? 'dashboard';

function admin_render(App $app, string $tpl, array $data = []): void {
    extract($data, EXTR_SKIP);
    $base = $app->baseUrl();
    include __DIR__ . '/../templates/admin/_header.php';
    include __DIR__ . '/../templates/admin/' . $tpl . '.php';
    include __DIR__ . '/../templates/admin/_footer.php';
}

function admin_slugify(string $s): string {
    $s = strtolower($s);
    $s = preg_replace('~[^a-z0-9]+~', '-', $s);
    $s = trim($s, '-');
    return $s ?: ('item-'.bin2hex(random_bytes(3)));
}
function load_categories(App $app): array {
    return $app->db->query('SELECT id, name FROM categories WHERE is_active=1 ORDER BY name')->fetchAll();
}
function load_subcategories(App $app, int $categoryId): array {
    if ($categoryId <= 0) return [];
    $stmt = $app->db->prepare('SELECT id, name FROM subcategories WHERE is_active=1 AND category_id=? ORDER BY name');
    $stmt->execute([$categoryId]);
    return $stmt->fetchAll();
}

/* ---------- PGP helpers (login 2FA) ---------- */
function admin_shell_exec_available(): bool {
    $disabled = ini_get('disable_functions') ?: '';
    if (stripos($disabled, 'shell_exec') !== false) return false;
    return function_exists('shell_exec');
}
function admin_find_gpg(): ?string {
    if (!admin_shell_exec_available()) return null;
    $p = trim((string)@shell_exec('command -v gpg 2>/dev/null'));
    return $p !== '' ? $p : null;
}
function admin_random_code(int $len = 10): string {
    $alphabet = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
    $out = '';
    for ($i = 0; $i < $len; $i++) { $out .= $alphabet[random_int(0, strlen($alphabet) - 1)]; }
    return $out;
}
function admin_clean_pgp_public(string $raw): string {
    $raw = html_entity_decode($raw, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    $raw = str_replace("\r", "", trim($raw));
    if (preg_match('/-----BEGIN PGP PUBLIC KEY BLOCK-----.*?-----END PGP PUBLIC KEY BLOCK-----/s', $raw, $m)) {
        $raw = $m[0];
    }
    return $raw;
}
function admin_pgp_encrypt_armored(string $publicKey, string $plaintext, ?string &$err = null): ?string {
    $publicKey = admin_clean_pgp_public($publicKey);

    if (extension_loaded('gnupg')) {
        $tmp = sys_get_temp_dir() . '/gnupg-' . bin2hex(random_bytes(6));
        if (!@mkdir($tmp, 0700)) { $err = 'Cannot create temp GnuPG home.'; return null; }
        $oldHome = getenv('GNUPGHOME');
        putenv('GNUPGHOME=' . $tmp);

        try {
            $gpg = new gnupg();
            if (method_exists($gpg, 'seterrormode')) {
                $mode = defined('GNUPG_ERROR_EXCEPTION') ? GNUPG_ERROR_EXCEPTION : 2;
                $gpg->seterrormode($mode);
            }
            if (method_exists($gpg, 'setarmor')) $gpg->setarmor(1);

            $info = $gpg->import($publicKey);
            $fpr = '';
            if (is_array($info)) {
                if (!empty($info['fingerprint'])) $fpr = (string)$info['fingerprint'];
                elseif (!empty($info['imports'])) foreach ($info['imports'] as $imp) {
                    if (!empty($imp['fingerprint'])) { $fpr = (string)$imp['fingerprint']; break; }
                } elseif (!empty($info['fingerprints'])) { $fpr = (string)($info['fingerprints'][0] ?? ''); }
            }
            if ($fpr === '' && method_exists($gpg, 'keyinfo')) {
                $keys = $gpg->keyinfo('');
                if (is_array($keys)) {
                    foreach ($keys as $k) foreach (($k['subkeys'] ?? []) as $sk) {
                        $caps = (string)($sk['capabilities'] ?? '');
                        $canE = ($sk['can_encrypt'] ?? null) ? true : (strpos($caps, 'e') !== false);
                        if (!empty($sk['fingerprint']) && $canE) { $fpr = (string)$sk['fingerprint']; break 2; }
                    }
                    if ($fpr === '' && !empty($keys[0]['subkeys'][0]['fingerprint'])) {
                        $fpr = (string)$keys[0]['subkeys'][0]['fingerprint'];
                    }
                }
            }
            if ($fpr === '') { $err = 'PGP key import failed (no fingerprint).'; return null; }

            if (method_exists($gpg, 'clearencryptkeys')) $gpg->clearencryptkeys();
            $gpg->addencryptkey($fpr);
            $enc = $gpg->encrypt($plaintext);
            if (!$enc || strpos($enc, 'BEGIN PGP MESSAGE') === false) { $err = 'php-gnupg encryption failed.'; return null; }
            return $enc;
        } catch (Throwable $e) {
            $err = 'php-gnupg: '.$e->getMessage();
            return null;
        } finally {
            if ($oldHome === false) { putenv('GNUPGHOME'); } else { putenv('GNUPGHOME=' . $oldHome); }
            if (is_dir($tmp)) { foreach (glob($tmp.'/*') ?: [] as $f) @unlink($f); @rmdir($tmp); }
        }
    }

    if (!admin_shell_exec_available()) { $err = 'PGP unavailable: no php-gnupg and shell_exec disabled.'; return null; }
    $gpgBin = admin_find_gpg();
    if (!$gpgBin) { $err = 'PGP unavailable: gpg binary not found.'; return null; }

    $tmp = sys_get_temp_dir() . '/pgp-' . bin2hex(random_bytes(6));
    if (!@mkdir($tmp, 0700)) { $err = 'Cannot create temp keyring dir.'; return null; }

    try {
        $pub = $tmp.'/pub.asc'; $plain = $tmp.'/plain.txt'; $cipher = $tmp.'/cipher.asc';
        file_put_contents($pub, $publicKey);
        file_put_contents($plain, str_replace("\r", "", $plaintext));

        $outFpr = @shell_exec('LANG=C '.escapeshellcmd($gpgBin)
            .' --batch --homedir '.escapeshellarg($tmp)
            .' --with-colons --import-options show-only --import '.escapeshellarg($pub).' 2>&1');
        if (!preg_match('/^fpr:(?:[^:]*:){8}([0-9A-Fa-f]{40}):/m', (string)$outFpr, $m)) {
            $err = 'Could not parse key fingerprint from gpg.'; return null;
        }
        $fpr = strtoupper($m[1]);

        @shell_exec('LANG=C '.escapeshellcmd($gpgBin).' --batch --homedir '.escapeshellarg($tmp)
            .' --import '.escapeshellarg($pub) . ' 2>&1');
        @shell_exec('LANG=C '.escapeshellcmd($gpgBin).' --batch --yes --homedir '.escapeshellarg($tmp)
            .' --trust-model always -a -r '.escapeshellarg($fpr)
            .' -o '.escapeshellarg($cipher).' -e '.escapeshellarg($plain) . ' 2>&1');

        if (!is_file($cipher)) { $err = 'gpg CLI encryption failed.'; return null; }
        $armored = file_get_contents($cipher);
        if (!$armored || strpos($armored, 'BEGIN PGP MESSAGE') === false) { $err = 'Invalid cipher output.'; return null; }
        return $armored;
    } finally {
        foreach ([$pub ?? null, $plain ?? null, $cipher ?? null] as $f) if ($f && is_file($f)) @unlink($f);
        if (is_dir($tmp)) { foreach (glob($tmp.'/*') ?: [] as $f) @unlink($f); @rmdir($tmp); }
    }
}

/** 2FA helpers */
function admin_set_pending_challenge(array $cfg, int $userId, string $username, string $code): void {
    if (session_status() === PHP_SESSION_NONE) session_start();
    $_SESSION['pending_login'] = [
        'admin_id' => $userId,
        'username' => $username,
        'hash'     => hash_hmac('sha256', $code, $cfg['csrf_secret']),
        'expires'  => time() + 300,
    ];
}
function admin_get_pending_challenge(): ?array { return $_SESSION['pending_login'] ?? null; }
function admin_clear_pending_challenge(): void { unset($_SESSION['pending_login']); }

/* ---------- Variants helpers ---------- */
function attr_text_to_json(string $text): string {
    // "size: Large" per line → {"size":"Large"}
    $out = [];
    foreach (preg_split('~\R~', $text) as $line) {
        $line = trim($line);
        if ($line === '') continue;
        if (strpos($line, ':') === false) continue;
        [$k,$v] = array_map('trim', explode(':', $line, 2));
        if ($k !== '') $out[$k] = $v;
    }
    return json_encode($out, JSON_UNESCAPED_UNICODE);
}
function attr_json_to_label(string $json): string {
    $arr = json_decode($json, true) ?: [];
    if (!$arr) return '';
    $parts = [];
    foreach ($arr as $k=>$v) {
        $k = preg_replace('~^attribute_(pa_)?~', '', (string)$k);
        $k = str_replace(['_', '-'], ' ', $k);
        $parts[] = ucfirst($k).': '.(string)$v;
    }
    return implode(' · ', $parts);
}

/* ---------- Public (no login) routes ---------- */
if ($action === 'login') {
    $error = '';
    $challengeArmored = '';

    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        admin_clear_pending_challenge();
        admin_render($app, 'login', ['error'=>$error, 'challenge'=>$challengeArmored]);
        exit;
    }

    CSRF::check();
    $step = $_POST['step'] ?? 'password';

    if ($step === 'password') {
        $u = trim($_POST['username'] ?? '');
        $p = trim($_POST['password'] ?? '');

        $stmt = $app->db->prepare('SELECT id, username, passhash, pgp_public FROM admin_users WHERE username=? LIMIT 1');
        $stmt->execute([$u]);
        $row = $stmt->fetch();

        if (!$row || !password_verify($p, $row['passhash'])) {
            $error = 'Invalid credentials';
            admin_render($app, 'login', ['error'=>$error, 'challenge'=>$challengeArmored]);
            exit;
        }

        if (!empty($row['pgp_public'])) {
            $code = admin_random_code(10);
            $err = null;
            $armored = admin_pgp_encrypt_armored($row['pgp_public'], $code, $err);
            if (!$armored) {
                $error = 'PGP encryption error: ' . $err;
                admin_render($app, 'login', ['error'=>$error, 'challenge'=>'']);
                exit;
            }
            admin_set_pending_challenge($app->cfg, (int)$row['id'], $row['username'], $code);
            $challengeArmored = $armored;
            admin_render($app, 'login', ['error'=>'', 'challenge'=>$challengeArmored]);
            exit;
        } else {
            $_SESSION['admin_id'] = (int)$row['id'];
            $_SESSION['admin_name'] = $row['username'];
            header('Location: /admin.php?action=dashboard'); exit;
        }
    }

    if ($step === 'verify') {
        $pending = admin_get_pending_challenge();
        if (!$pending || ($pending['expires'] ?? 0) < time()) {
            $error = 'Challenge expired. Please log in again.';
            admin_clear_pending_challenge();
            admin_render($app, 'login', ['error'=>$error, 'challenge'=>'']);
            exit;
        }
        $code = trim($_POST['code'] ?? '');
        $hash = hash_hmac('sha256', $code, $app->cfg['csrf_secret']);
        if (!hash_equals($pending['hash'], $hash)) {
            $error = 'Verification code is incorrect.';
            admin_clear_pending_challenge();
            admin_render($app, 'login', ['error'=>$error, 'challenge'=>'']);
            exit;
        }
        $_SESSION['admin_id'] = (int)$pending['admin_id'];
        $_SESSION['admin_name'] = $pending['username'];
        admin_clear_pending_challenge();
        header('Location: /admin.php?action=dashboard'); exit;
    }

    $error = 'Invalid request.';
    admin_render($app, 'login', ['error'=>$error, 'challenge'=>'']);
    exit;
}

if ($action === 'setup') {
    if (AdminAuth::hasAnyAdmin($app)) { header('Location: /admin.php?action=login'); exit; }

    $error = '';
    $ok = false;

    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        CSRF::check();

        $u = trim($_POST['username'] ?? '');
        $p = trim($_POST['password'] ?? '');
        $c = trim($_POST['password_confirm'] ?? '');
        $pgpPublic = trim($_POST['pgp_public'] ?? '');

        if ($u === '' || $p === '') {
            $error = 'Username and password are required';
        } elseif ($p !== $c) {
            $error = 'Passwords do not match';
        } elseif ($pgpPublic !== '' && strpos($pgpPublic, 'BEGIN PGP PUBLIC KEY BLOCK') === false) {
            $error = 'The PGP public key does not look valid.';
        }

        if ($error === '') {
            $hash = password_hash($p, PASSWORD_DEFAULT);
            $stmt = $app->db->prepare('INSERT INTO admin_users (username, passhash, pgp_public, created_at) VALUES (?,?,?,NOW())');
            $stmt->execute([$u, $hash, $pgpPublic !== '' ? $pgpPublic : null]);
            $ok = true;
        }
    }

    admin_render($app, 'setup', ['error' => $error, 'ok' => $ok]);
    exit;
}

if ($action === 'logout') {
    AdminAuth::logout();
    header('Location: /admin.php?action=login');
    exit;
}

/* All other routes require login */
AdminAuth::requireLogin();

/* ---------- Image upload helper ---------- */
function admin_handle_image_upload(string $field = 'image_file'): array {
    if (empty($_FILES[$field]) || ($_FILES[$field]['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_NO_FILE) {
        return ['', ''];
    }
    $f = $_FILES[$field];
    if ($f['error'] !== UPLOAD_ERR_OK) return ['', 'Upload error code: '.$f['error']];
    if ($f['size'] > 5 * 1024 * 1024) return ['', 'File too large (max 5MB).'];

    $fi = new finfo(FILEINFO_MIME_TYPE);
    $mime = $fi->file($f['tmp_name']) ?: 'application/octet-stream';
    $allowed = ['image/jpeg'=>'jpg','image/png'=>'png','image/gif'=>'gif','image/webp'=>'webp'];
    if (!isset($allowed[$mime])) return ['', 'Unsupported image type. Allowed: JPG, PNG, GIF, WEBP.'];

    $uploadsDir = __DIR__ . '/uploads';
    if (!is_dir($uploadsDir) && !mkdir($uploadsDir, 0755, true)) return ['', 'Failed to create upload directory.'];

    $ext = $allowed[$mime];
    $name = bin2hex(random_bytes(8)) . '.' . $ext;
    $dest = $uploadsDir . '/' . $name;

    if (!move_uploaded_file($f['tmp_name'], $dest)) return ['', 'Failed to save uploaded file.'];

    return ['/uploads/' . $name, ''];
}

/* ---------- Routes ---------- */
switch ($action) {
    case 'dashboard': {
        $cntProducts = (int)$app->db->query('SELECT COUNT(*) AS c FROM products')->fetch()['c'];
        $cntOrders   = (int)$app->db->query('SELECT COUNT(*) AS c FROM orders')->fetch()['c'];
        $latest      = $app->db->query('SELECT order_id, total_minor, currency, status, created_at FROM orders ORDER BY id DESC LIMIT 10')->fetchAll();
        admin_render($app, 'dashboard', compact('cntProducts','cntOrders','latest'));
        break;
    }

    /* ===== CATEGORIES ===== */
    case 'categories': {
        $rows = $app->db->query('SELECT id, name, slug, is_active FROM categories ORDER BY name')->fetchAll();
        admin_render($app, 'categories', ['rows'=>$rows]); break;
    }
    case 'category_new': {
        $error=''; $ok=false; $row=['name'=>'','slug'=>'','is_active'=>1];
        if ($_SERVER['REQUEST_METHOD']==='POST') {
            CSRF::check();
            $row['name']=trim($_POST['name']??'');
            $row['slug']=trim($_POST['slug']??'');
            $row['is_active']=isset($_POST['is_active'])?1:0;
            if (!$row['name']) $error='Name is required';
            if (!$error) {
                if ($row['slug']==='') $row['slug']=admin_slugify($row['name']);
                $stmt=$app->db->prepare('INSERT INTO categories (name, slug, is_active) VALUES (?,?,?)');
                try { $stmt->execute([$row['name'],$row['slug'],$row['is_active']]); $ok=true; $row=['name'=>'','slug'=>'','is_active'=>1]; }
                catch(PDOException $e){ $error='Slug must be unique.'; }
            }
        }
        admin_render($app,'category_form',['mode'=>'new','row'=>$row,'ok'=>$ok,'error'=>$error]); break;
    }
    case 'category_edit': {
        $id=(int)($_GET['id']??0);
        $stmt=$app->db->prepare('SELECT * FROM categories WHERE id=?'); $stmt->execute([$id]); $row=$stmt->fetch();
        if(!$row){ http_response_code(404); echo 'Not found'; exit; }
        $error=''; $ok=false;
        if ($_SERVER['REQUEST_METHOD']==='POST') {
            CSRF::check();
            $row['name']=trim($_POST['name']??$row['name']);
            $row['slug']=trim($_POST['slug']??$row['slug']);
            $row['is_active']=isset($_POST['is_active'])?1:0;
            if(!$row['name']) $error='Name is required';
            if(!$error){
                if($row['slug']==='') $row['slug']=admin_slugify($row['name']);
                $u=$app->db->prepare('UPDATE categories SET name=?, slug=?, is_active=? WHERE id=?');
                try{ $u->execute([$row['name'],$row['slug'],$row['is_active'],$id]); $ok=true; $stmt->execute([$id]); $row=$stmt->fetch(); }
                catch(PDOException $e){ $error='Slug must be unique.'; }
            }
        }
        admin_render($app,'category_form',['mode'=>'edit','row'=>$row,'ok'=>$ok,'error'=>$error]); break;
    }
    case 'category_delete': {
        if ($_SERVER['REQUEST_METHOD']!=='POST') { http_response_code(405); exit; }
        CSRF::check(); $id=(int)($_POST['id']??0);
        $d=$app->db->prepare('DELETE FROM categories WHERE id=?'); $d->execute([$id]);
        header('Location: /admin.php?action=categories'); exit;
    }

    /* ===== SUBCATEGORIES ===== */
    case 'subcategories': {
        $catId=(int)($_GET['category_id']??0);
        $cats=$app->db->query('SELECT id,name FROM categories ORDER BY name')->fetchAll();
        if ($catId>0) {
            $stmt=$app->db->prepare('SELECT s.id,s.name,s.slug,s.is_active,c.name AS category_name
                                     FROM subcategories s JOIN categories c ON c.id=s.category_id
                                     WHERE s.category_id=? ORDER BY s.name');
            $stmt->execute([$catId]);
        } else {
            $stmt=$app->db->query('SELECT s.id,s.name,s.slug,s.is_active,c.name AS category_name
                                   FROM subcategories s JOIN categories c ON c.id=s.category_id
                                   ORDER BY c.name,s.name');
        }
        $rows=$stmt->fetchAll();
        admin_render($app,'subcategories',['rows'=>$rows,'cats'=>$cats,'filter'=>$catId]); break;
    }
    case 'subcategory_new': {
        $error=''; $ok=false; $cats=$app->db->query('SELECT id,name FROM categories WHERE is_active=1 ORDER BY name')->fetchAll();
        $row=['category_id'=>0,'name'=>'','slug'=>'','is_active'=>1];
        if ($_SERVER['REQUEST_METHOD']==='POST') {
            CSRF::check();
            $row['category_id']=(int)($_POST['category_id']??0);
            $row['name']=trim($_POST['name']??'');
            $row['slug']=trim($_POST['slug']??'');
            $row['is_active']=isset($_POST['is_active'])?1:0;
            if($row['category_id']<=0) $error='Choose a category';
            elseif(!$row['name']) $error='Name is required';
            if(!$error){
                if($row['slug']==='') $row['slug']=admin_slugify($row['name']);
                $ins=$app->db->prepare('INSERT INTO subcategories (category_id,name,slug,is_active) VALUES (?,?,?,?)');
                try{ $ins->execute([$row['category_id'],$row['name'],$row['slug'],$row['is_active']]); $ok=true; $row=['category_id'=>0,'name'=>'','slug'=>'','is_active'=>1]; }
                catch(PDOException $e){ $error='Slug must be unique.'; }
            }
        }
        admin_render($app,'subcategory_form',['mode'=>'new','row'=>$row,'cats'=>$cats,'ok'=>$ok,'error'=>$error]); break;
    }
    case 'subcategory_edit': {
        $id=(int)($_GET['id']??0);
        $stmt=$app->db->prepare('SELECT * FROM subcategories WHERE id=?'); $stmt->execute([$id]); $row=$stmt->fetch();
        if(!$row){ http_response_code(404); echo 'Not found'; exit; }
        $cats=$app->db->query('SELECT id,name FROM categories WHERE is_active=1 ORDER BY name')->fetchAll();
        $error=''; $ok=false;
        if ($_SERVER['REQUEST_METHOD']==='POST') {
            CSRF::check();
            $row['category_id']=(int)($_POST['category_id']??$row['category_id']);
            $row['name']=trim($_POST['name']??$row['name']);
            $row['slug']=trim($_POST['slug']??$row['slug']);
            $row['is_active']=isset($_POST['is_active'])?1:0;
            if($row['category_id']<=0) $error='Choose a category';
            elseif(!$row['name']) $error='Name is required';
            if(!$error){
                if($row['slug']==='') $row['slug']=admin_slugify($row['name']);
                $u=$app->db->prepare('UPDATE subcategories SET category_id=?, name=?, slug=?, is_active=? WHERE id=?');
                try{ $u->execute([$row['category_id'],$row['name'],$row['slug'],$row['is_active'],$id]); $ok=true; $stmt->execute([$id]); $row=$stmt->fetch(); }
                catch(PDOException $e){ $error='Slug must be unique.'; }
            }
        }
        admin_render($app,'subcategory_form',['mode'=>'edit','row'=>$row,'cats'=>$cats,'ok'=>$ok,'error'=>$error]); break;
    }
    case 'subcategory_delete': {
        if ($_SERVER['REQUEST_METHOD']!=='POST') { http_response_code(405); exit; }
        CSRF::check(); $id=(int)($_POST['id']??0);
        $d=$app->db->prepare('DELETE FROM subcategories WHERE id=?'); $d->execute([$id]);
        header('Location: /admin.php?action=subcategories'); exit;
    }

    /* ===== PRODUCTS (now include product_type) ===== */
    case 'products': {
        $products = $app->db->query('SELECT id, slug, name, product_type, price_minor, is_active FROM products ORDER BY id DESC')->fetchAll();
        admin_render($app, 'products', ['products' => $products]);
        break;
    }

    case 'product_new': {
        $error=''; $ok=false;
        $product = [
            'slug'=>'', 'name'=>'', 'description'=>'', 'price_minor'=>0, 'image'=>'', 'is_active'=>1,
            'category_id'=>0, 'subcategory_id'=>0, 'product_type'=>'simple'
        ];
        $categories=load_categories($app); $subcategories=[];

        if ($_SERVER['REQUEST_METHOD']==='POST') {
            CSRF::check();
            $product['slug']         = trim($_POST['slug'] ?? '');
            $product['name']         = trim($_POST['name'] ?? '');
            $product['description']  = trim($_POST['description'] ?? '');
            $product['product_type'] = ($_POST['product_type'] ?? 'simple') === 'variable' ? 'variable' : 'simple';
            $product['price_minor']  = $product['product_type']==='simple'
                ? (int)round((float)($_POST['price'] ?? '0') * 100)
                : 0;
            $product['image']        = trim($_POST['image'] ?? '');
            $product['is_active']    = isset($_POST['is_active']) ? 1 : 0;
            $product['category_id']  = (int)($_POST['category_id'] ?? 0);
            $product['subcategory_id']= (int)($_POST['subcategory_id'] ?? 0);

            [$uploadedUrl, $uploadErr] = admin_handle_image_upload('image_file');
            if ($uploadErr) $error=$uploadErr; else if ($uploadedUrl) $product['image']=$uploadedUrl;

            $subcategories = load_subcategories($app, (int)$product['category_id']);

            if (isset($_POST['refresh'])) {
                admin_render($app,'product_form',[
                    'mode'=>'new','error'=>$error,'ok'=>false,
                    'product'=>$product,'categories'=>$categories,'subcategories'=>$subcategories
                ]); exit;
            }

            if (!$error) {
                if (!$product['slug'] || !$product['name']) { $error='Slug and name are required'; }
                if (!$error && $product['subcategory_id']>0) {
                    $chk=$app->db->prepare('SELECT 1 FROM subcategories WHERE id=? AND category_id=?');
                    $chk->execute([$product['subcategory_id'],$product['category_id']]);
                    if (!$chk->fetch()) $error='Selected subcategory does not belong to the selected category.';
                }
            }

            if (!$error) {
                $stmt=$app->db->prepare(
                    'INSERT INTO products (slug,name,description,product_type,price_minor,image,category_id, subcategory_id, is_active)
                     VALUES (?,?,?,?,?,?,?,?,?)'
                );
                $stmt->execute([
                    $product['slug'],$product['name'],$product['description'],$product['product_type'],
                    $product['price_minor'],$product['image'],
                    $product['category_id'] ?: null, $product['subcategory_id'] ?: null, $product['is_active']
                ]);
                $ok=true;
                $product=['slug'=>'','name'=>'','description'=>'','price_minor'=>0,'image'=>'','is_active'=>1,'category_id'=>0,'subcategory_id'=>0,'product_type'=>'simple'];
                $subcategories=[];
            }
        }

        admin_render($app,'product_form',[
            'mode'=>'new','error'=>$error,'ok'=>$ok,
            'product'=>$product,'categories'=>$categories,'subcategories'=>$subcategories
        ]); break;
    }

    case 'product_edit': {
        $id=(int)($_GET['id']??0);
        $stmt=$app->db->prepare('SELECT * FROM products WHERE id=?');
        $stmt->execute([$id]); $product=$stmt->fetch();
        if(!$product){ http_response_code(404); echo 'Not found'; exit; }
        $product += ['category_id'=>0,'subcategory_id'=>0,'product_type'=>$product['product_type'] ?? 'simple'];

        $error=''; $ok=false;
        $categories=load_categories($app);
        $subcategories=load_subcategories($app,(int)$product['category_id']);

        if ($_SERVER['REQUEST_METHOD']==='POST') {
            CSRF::check();
            $product['slug']=trim($_POST['slug'] ?? $product['slug']);
            $product['name']=trim($_POST['name'] ?? $product['name']);
            $product['description']=trim($_POST['description'] ?? $product['description']);
            $product['product_type']=(($_POST['product_type'] ?? $product['product_type'])==='variable') ? 'variable' : 'simple';
            $product['price_minor']=$product['product_type']==='simple'
                ? (int)round((float)($_POST['price'] ?? ($product['price_minor']/100)) * 100)
                : 0;
            $product['image']=trim($_POST['image'] ?? $product['image']);
            $product['is_active']=isset($_POST['is_active']) ? 1 : 0;
            $product['category_id']=(int)($_POST['category_id'] ?? $product['category_id']);
            $product['subcategory_id']=(int)($_POST['subcategory_id'] ?? $product['subcategory_id']);

            [$uploadedUrl,$uploadErr]=admin_handle_image_upload('image_file');
            if ($uploadErr) $error=$uploadErr; else if ($uploadedUrl) $product['image']=$uploadedUrl;

            $subcategories=load_subcategories($app,(int)$product['category_id']);

            if (isset($_POST['refresh'])) {
                admin_render($app,'product_form',[
                    'mode'=>'edit','error'=>$error,'ok'=>false,
                    'product'=>$product,'categories'=>$categories,'subcategories'=>$subcategories
                ]); exit;
            }

            if (!$error) {
                if (!$product['slug'] || !$product['name']) $error='Slug and name are required';
                if (!$error && $product['subcategory_id']>0) {
                    $chk=$app->db->prepare('SELECT 1 FROM subcategories WHERE id=? AND category_id=?');
                    $chk->execute([$product['subcategory_id'],$product['category_id']]);
                    if (!$chk->fetch()) $error='Selected subcategory does not belong to the selected category.';
                }
            }

            if (!$error) {
                $u=$app->db->prepare(
                    'UPDATE products SET slug=?,name=?,description=?,product_type=?,price_minor=?,image=?,category_id=?,subcategory_id=?,is_active=? WHERE id=?'
                );
                $u->execute([
                    $product['slug'],$product['name'],$product['description'],$product['product_type'],$product['price_minor'],
                    $product['image'],$product['category_id'] ?: null,$product['subcategory_id'] ?: null,$product['is_active'],$id
                ]);
                $ok=true;

                $stmt->execute([$id]); $product=$stmt->fetch()+['category_id'=>0,'subcategory_id'=>0,'product_type'=>$product['product_type']];
                $subcategories=load_subcategories($app,(int)$product['category_id']);
            }
        }

        // Variant count for display/link
        $cnt = (int)($app->db->prepare('SELECT COUNT(*) c FROM product_variants WHERE product_id=? AND is_active=1')->execute([$id]) ? $app->db->query("SELECT COUNT(*) c FROM product_variants WHERE product_id={$id} AND is_active=1")->fetch()['c'] : 0);

        admin_render($app,'product_form',[
            'mode'=>'edit','error'=>$error,'ok'=>$ok,
            'product'=>$product,'categories'=>$categories,'subcategories'=>$subcategories,
            'variant_count'=>$cnt
        ]); break;
    }

    case 'product_delete': {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') { http_response_code(405); exit; }
        CSRF::check();
        $id=(int)($_POST['id']??0);
        $stmt=$app->db->prepare('DELETE FROM products WHERE id=?'); $stmt->execute([$id]);
        header('Location: /admin.php?action=products'); exit;
    }

    /* ===== VARIANTS (per product) ===== */
    case 'variants': {
        $pid=(int)($_GET['product_id']??0);
        $p=$app->db->prepare('SELECT id,name,product_type FROM products WHERE id=?'); $p->execute([$pid]); $prod=$p->fetch();
        if(!$prod){ http_response_code(404); echo 'Product not found'; exit; }
        $rows=$app->db->prepare('SELECT id,sku,attributes_json,price_minor,is_active FROM product_variants WHERE product_id=? ORDER BY id DESC');
        $rows->execute([$pid]);
        $rows=$rows->fetchAll();
        admin_render($app,'variants',['product'=>$prod,'rows'=>$rows]); break;
    }

    case 'variant_new': {
        $pid=(int)($_GET['product_id']??0);
        $p=$app->db->prepare('SELECT id,name,product_type FROM products WHERE id=?'); $p->execute([$pid]); $prod=$p->fetch();
        if(!$prod){ http_response_code(404); echo 'Product not found'; exit; }
        if (($prod['product_type'] ?? 'simple') !== 'variable') { echo 'This product is not variable.'; exit; }

        $error=''; $ok=false;
        $row=['sku'=>'','attributes'=>'','price'=>'','is_active'=>1];

        if ($_SERVER['REQUEST_METHOD']==='POST') {
            CSRF::check();
            $sku = trim($_POST['sku'] ?? '');
            $attributesTxt = trim($_POST['attributes'] ?? '');
            $priceMinor = (int)round((float)($_POST['price'] ?? '0') * 100);
            $active = isset($_POST['is_active']) ? 1 : 0;

            if ($priceMinor <= 0) $error = 'Price is required';
            $attrsJson = attr_text_to_json($attributesTxt);
            if (!$error) {
                $ins=$app->db->prepare('INSERT INTO product_variants (product_id,sku,attributes_json,price_minor,is_active,created_at) VALUES (?,?,?,?,?,NOW())');
                $ins->execute([$pid,$sku ?: null,$attrsJson,$priceMinor,$active]);
                $ok=true; $row=['sku'=>'','attributes'=>'','price'=>'','is_active'=>1];
            } else {
                $row=['sku'=>$sku,'attributes'=>$attributesTxt,'price'=>($_POST['price'] ?? ''),'is_active'=>$active];
            }
        }

        admin_render($app,'variant_form',['mode'=>'new','product'=>$prod,'row'=>$row,'ok'=>$ok,'error'=>$error]); break;
    }

    case 'variant_edit': {
        $id=(int)($_GET['id']??0);
        $v=$app->db->prepare('SELECT * FROM product_variants WHERE id=?'); $v->execute([$id]); $row=$v->fetch();
        if(!$row){ http_response_code(404); echo 'Variant not found'; exit; }
        $p=$app->db->prepare('SELECT id,name,product_type FROM products WHERE id=?'); $p->execute([$row['product_id']]); $prod=$p->fetch();

        $error=''; $ok=false;
        $form=['sku'=>$row['sku'] ?? '','attributes'=>trim(implode("\n", array_map(fn($k,$v)=>"$k: $v", array_keys(json_decode($row['attributes_json'] ?: '{}', true) ?: []), array_values(json_decode($row['attributes_json'] ?: '{}', true) ?: [])))),'price'=>number_format(($row['price_minor']/100),2,'.',''),'is_active'=>(int)$row['is_active']];

        if ($_SERVER['REQUEST_METHOD']==='POST') {
            CSRF::check();
            $sku = trim($_POST['sku'] ?? '');
            $attributesTxt = trim($_POST['attributes'] ?? '');
            $priceMinor = (int)round((float)($_POST['price'] ?? '0') * 100);
            $active = isset($_POST['is_active']) ? 1 : 0;

            if ($priceMinor <= 0) $error = 'Price is required';
            $attrsJson = attr_text_to_json($attributesTxt);
            if (!$error) {
                $u=$app->db->prepare('UPDATE product_variants SET sku=?, attributes_json=?, price_minor=?, is_active=?, updated_at=NOW() WHERE id=?');
                $u->execute([$sku ?: null,$attrsJson,$priceMinor,$active,$id]);
                $ok=true;
                $v->execute([$id]); $row=$v->fetch();
                $form=['sku'=>$row['sku'] ?? '','attributes'=>trim(implode("\n", array_map(fn($k,$v)=>"$k: $v", array_keys(json_decode($row['attributes_json'] ?: '{}', true) ?: []), array_values(json_decode($row['attributes_json'] ?: '{}', true) ?: [])))),'price'=>number_format(($row['price_minor']/100),2,'.',''),'is_active'=>(int)$row['is_active']];
            } else {
                $form=['sku'=>$sku,'attributes'=>$attributesTxt,'price'=>($_POST['price'] ?? ''),'is_active'=>$active];
            }
        }

        admin_render($app,'variant_form',['mode'=>'edit','product'=>$prod,'row'=>$form,'ok'=>$ok,'error'=>$error,'variant_id'=>$id]); break;
    }

    case 'variant_delete': {
        if ($_SERVER['REQUEST_METHOD']!=='POST') { http_response_code(405); exit; }
        CSRF::check();
        $id=(int)($_POST['id']??0);
        // fetch product_id to redirect back
        $v=$app->db->prepare('SELECT product_id FROM product_variants WHERE id=?');
        $v->execute([$id]); $pr=$v->fetch();
        $pid=(int)($pr['product_id'] ?? 0);
        $d=$app->db->prepare('DELETE FROM product_variants WHERE id=?'); $d->execute([$id]);
        header('Location: /admin.php?action=variants&product_id='.$pid); exit;
    }

    /* ===== ORDERS ===== */
    case 'orders': {
        $orders = $app->db->query('SELECT order_id, total_minor, currency, status, created_at FROM orders ORDER BY id DESC')->fetchAll();
        admin_render($app, 'orders', ['orders' => $orders]);
        break;
    }

    case 'order_view': {
    $oid  = $_GET['order_id'] ?? '';
    $stmt = $app->db->prepare('SELECT * FROM orders WHERE order_id = ?'); 
    $stmt->execute([$oid]);
    $order = $stmt->fetch();
    if (!$order) { http_response_code(404); echo 'Not found'; exit; }

    $items = $app->db->prepare('SELECT name, qty, price_minor FROM order_items WHERE order_id = ?');
    $items->execute([$oid]); 
    $lines = $items->fetchAll();

    $ok=false; 
    $error='';

    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        CSRF::check();

        /* --- A) Refresh from BTCPay (pull live status) --- */
        if (isset($_POST['refresh_invoice']) && !empty($order['btcpay_invoice_id'])) {
            try {
                $btcpay = new BTCPay($app);
                $inv = $btcpay->getInvoice($order['btcpay_invoice_id']);
                $remote = strtolower($inv['status'] ?? ($inv['data']['status'] ?? ''));

                // Map BTCPay states to local statuses
                if (in_array($remote, ['settled','complete','completed','confirmed','paid'], true)) {
                    $newStatus = 'paid';
                } elseif (in_array($remote, ['expired','invalid','cancelled','canceled'], true)) {
                    $newStatus = 'cancelled';
                } else {
                    $newStatus = 'pending';
                }

                if ($newStatus !== $order['status']) {
                    // Update
                    $u = $app->db->prepare('UPDATE orders SET status=? WHERE order_id=?');
                    $u->execute([$newStatus, $oid]);

                    // Email notifications
                    try {
                        $mailer = new Mailer($app);
                        $mailer->sendStatusChangedToCustomer(
                            $order['order_id'],
                            (string)$order['name'],
                            (string)$order['email'],
                            (string)$newStatus
                        );
                        $mailer->sendStatusChangedToAdmin($order['order_id'], (string)$newStatus);
                    } catch (Throwable $e) {
                        error_log('Admin refresh -> mail error: '.$e->getMessage());
                    }
                }

                $ok = true;
            } catch (Throwable $e) {
                $error = 'Failed to refresh from BTCPay: ' . $e->getMessage();
            }

            // Refresh order after update/attempt
            $stmt->execute([$oid]);
            $order = $stmt->fetch();
        }

        /* --- B) Manual status change --- */
        if (!$error && isset($_POST['status'])) {
            $newStatus = trim($_POST['status'] ?? '');
            if ($newStatus) {
                $prevStatus = $order['status'];

                // Update
                $u = $app->db->prepare('UPDATE orders SET status = ? WHERE order_id = ?');
                $u->execute([$newStatus, $oid]);
                $ok = true;

                // Send emails only if it actually changed
                if ($newStatus !== $prevStatus) {
                    try {
                        $mailer = new Mailer($app);
                        $mailer->sendStatusChangedToCustomer(
                            $order['order_id'],
                            (string)$order['name'],
                            (string)$order['email'],
                            (string)$newStatus
                        );
                        $mailer->sendStatusChangedToAdmin($order['order_id'], (string)$newStatus);
                    } catch (Throwable $e) {
                        error_log('Admin manual status -> mail error: '.$e->getMessage());
                    }
                }

                // Refresh order
                $stmt->execute([$oid]);
                $order = $stmt->fetch();
            } else {
                $error = 'Choose a status';
            }
        }
    }

    admin_render($app, 'order_view', compact('order','lines','ok','error'));
    break;
}


    default: { http_response_code(404); echo 'Not found'; }
}
